import { NextResponse } from "next/server";
import { deleteUser, updateUser } from "@/lib/userRepository";

type Body = {
  name?: string;
  email?: string;
  role?: string;
};

function parseIdFromParams(param: string | string[] | undefined): number | null {
  const source = Array.isArray(param) ? param[0] : param;
  const trimmed = source?.trim();

  if (!trimmed) {
    return null;
  }

  const parsed = Number(trimmed);

  if (!Number.isInteger(parsed) || parsed < 1) {
    return null;
  }

  return parsed;
}

function parseId(
  request: Request,
  params: { id: string | string[] }
): number | null {
  const idFromParams = parseIdFromParams(params.id);

  if (idFromParams !== null) {
    return idFromParams;
  }

  const url = new URL(request.url);
  const segments = url.pathname.split("/").filter(Boolean);
  const lastSegment = segments.pop();

  return parseIdFromParams(lastSegment);
}

function validateUserPayload(body: Body) {
  const errors: string[] = [];

  if (!body.name?.trim()) {
    errors.push("名前は必須です。");
  }

  if (!body.email?.trim()) {
    errors.push("メールアドレスは必須です。");
  } else {
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (!emailRegex.test(body.email)) {
      errors.push("メールアドレスの形式が正しくありません。");
    }
  }

  if (!body.role?.trim()) {
    errors.push("役割は必須です。");
  }

  return errors;
}

export async function PUT(
  request: Request,
  context: { params: Promise<{ id: string | string[] }> }
) {
  const params = await context.params;
  const id = parseId(request, params);

  if (id === null) {
    return NextResponse.json(
      { message: "不正なユーザーIDです。" },
      { status: 400 }
    );
  }

  try {
    const body = (await request.json()) as Body;
    const errors = validateUserPayload(body);

    if (errors.length) {
      return NextResponse.json({ message: errors.join("\n") }, { status: 400 });
    }

    const user = await updateUser(id, {
      name: body.name!.trim(),
      email: body.email!.trim(),
      role: body.role!.trim(),
    });

    return NextResponse.json(user);
  } catch (error) {
    console.error(error);

    if (error instanceof Error && error.message === "USER_NOT_FOUND") {
      return NextResponse.json(
        { message: "指定されたユーザーが見つかりません。" },
        { status: 404 }
      );
    }

    const message =
      error instanceof Error && error.message.includes("ER_DUP_ENTRY")
        ? "指定されたメールアドレスは既に使用されています。"
        : "ユーザーの更新に失敗しました。";

    return NextResponse.json({ message }, { status: 500 });
  }
}

export async function DELETE(
  request: Request,
  context: { params: Promise<{ id: string | string[] }> }
) {
  const params = await context.params;
  const id = parseId(request, params);

  if (id === null) {
    return NextResponse.json(
      { message: "不正なユーザーIDです。" },
      { status: 400 }
    );
  }

  try {
    await deleteUser(id);
    return NextResponse.json({ success: true });
  } catch (error) {
    console.error(error);

    if (error instanceof Error && error.message === "USER_NOT_FOUND") {
      return NextResponse.json(
        { message: "指定されたユーザーが見つかりません。" },
        { status: 404 }
      );
    }

    return NextResponse.json(
      { message: "ユーザーの削除に失敗しました。" },
      { status: 500 }
    );
  }
}
